import { CommandLinker } from '@jupyterlab/apputils';
import { DocumentRegistry } from '@jupyterlab/docregistry';
import { ServiceManager } from '@jupyterlab/services';
import { ContextMenuSvg } from '@jupyterlab/ui-components';
import { Application, IPlugin } from '@lumino/application';
import { Token } from '@lumino/coreutils';
import { ISignal } from '@lumino/signaling';
import { Widget } from '@lumino/widgets';
/**
 * The type for all JupyterFrontEnd application plugins.
 *
 * @typeparam T - The type that the plugin `provides` upon being activated.
 *
 * @typeparam U - The type of the application shell.
 *
 * @typeparam V - The type that defines the application formats.
 */
export type JupyterFrontEndPlugin<T, U extends JupyterFrontEnd.IShell = JupyterFrontEnd.IShell, V extends string = 'desktop' | 'mobile'> = IPlugin<JupyterFrontEnd<U, V>, T>;
/**
 * The base Jupyter front-end application class.
 *
 * @typeparam `T` - The `shell` type. Defaults to `JupyterFrontEnd.IShell`.
 *
 * @typeparam `U` - The type for supported format names. Defaults to `string`.
 *
 * #### Notes
 * This type is useful as a generic application against which front-end plugins
 * can be authored. It inherits from the Lumino `Application`.
 */
export declare abstract class JupyterFrontEnd<T extends JupyterFrontEnd.IShell = JupyterFrontEnd.IShell, U extends string = 'desktop' | 'mobile'> extends Application<T> {
    /**
     * Construct a new JupyterFrontEnd object.
     */
    constructor(options: JupyterFrontEnd.IOptions<T>);
    /**
     * The name of this Jupyter front-end application.
     */
    abstract readonly name: string;
    /**
     * A namespace/prefix plugins may use to denote their provenance.
     */
    abstract readonly namespace: string;
    /**
     * The version of this Jupyter front-end application.
     */
    abstract readonly version: string;
    /**
     * The command linker used by the application.
     */
    readonly commandLinker: CommandLinker;
    /**
     * The application context menu.
     */
    readonly contextMenu: ContextMenuSvg;
    /**
     * The document registry instance used by the application.
     */
    readonly docRegistry: DocumentRegistry;
    /**
     * Promise that resolves when state is first restored.
     */
    readonly restored: Promise<void>;
    /**
     * The service manager used by the application.
     */
    readonly serviceManager: ServiceManager.IManager;
    /**
     * The application form factor, e.g., `desktop` or `mobile`.
     */
    get format(): U;
    set format(format: U);
    /**
     * A signal that emits when the application form factor changes.
     */
    get formatChanged(): ISignal<this, U>;
    /**
     * Walks up the DOM hierarchy of the target of the active `contextmenu`
     * event, testing each HTMLElement ancestor for a user-supplied function. This can
     * be used to find an HTMLElement on which to operate, given a context menu click.
     *
     * @param fn - a function that takes an `HTMLElement` and returns a
     *   boolean for whether it is the element the requester is seeking.
     *
     * @returns an HTMLElement or undefined, if none is found.
     */
    contextMenuHitTest(fn: (node: HTMLElement) => boolean): HTMLElement | undefined;
    /**
     * A method invoked on a document `'contextmenu'` event.
     */
    protected evtContextMenu(event: MouseEvent): void;
    private _contextMenuEvent;
    private _format;
    private _formatChanged;
}
/**
 * The namespace for `JupyterFrontEnd` class statics.
 */
export declare namespace JupyterFrontEnd {
    /**
     * The options used to initialize a JupyterFrontEnd.
     */
    interface IOptions<T extends IShell = IShell, U = any> extends Application.IOptions<T> {
        /**
         * The document registry instance used by the application.
         */
        docRegistry?: DocumentRegistry;
        /**
         * The command linker used by the application.
         */
        commandLinker?: CommandLinker;
        /**
         * The service manager used by the application.
         */
        serviceManager?: ServiceManager.IManager;
        /**
         * Promise that resolves when state is first restored, returning layout
         * description.
         */
        restored?: Promise<U>;
    }
    /**
     * A minimal shell type for Jupyter front-end applications.
     */
    interface IShell extends Widget {
        /**
         * Activates a widget inside the application shell.
         *
         * @param id - The ID of the widget being activated.
         */
        activateById(id: string): void;
        /**
         * Add a widget to the application shell.
         *
         * @param widget - The widget being added.
         *
         * @param area - Optional region in the shell into which the widget should
         * be added.
         *
         * @param options - Optional flags the shell might use when opening the
         * widget, as defined in the `DocumentRegistry`.
         */
        add(widget: Widget, area?: string, options?: DocumentRegistry.IOpenOptions): void;
        /**
         * The focused widget in the application shell.
         *
         * #### Notes
         * Different shell implementations have latitude to decide what "current"
         * or "focused" mean, depending on their user interface characteristics.
         */
        readonly currentWidget: Widget | null;
        /**
         * Returns an iterator for the widgets inside the application shell.
         *
         * @param area - Optional regions in the shell whose widgets are iterated.
         */
        widgets(area?: string): IterableIterator<Widget>;
    }
    /**
     * Is JupyterLab in document mode?
     *
     * @param path - Full URL of JupyterLab
     * @param paths - The current IPaths object hydrated from PageConfig.
     */
    function inDocMode(path: string, paths: IPaths): boolean;
    /**
     * The application paths dictionary token.
     */
    const IPaths: Token<IPaths>;
    /**
     * An interface for URL and directory paths used by a Jupyter front-end.
     */
    interface IPaths {
        /**
         * The urls used by the application.
         */
        readonly urls: {
            readonly base: string;
            readonly notFound?: string;
            readonly app: string;
            readonly doc: string;
            readonly static: string;
            readonly settings: string;
            readonly themes: string;
            readonly translations: string;
            readonly hubPrefix?: string;
            readonly hubHost?: string;
            readonly hubUser?: string;
            readonly hubServerName?: string;
        };
        /**
         * The server directories used by the application, for user information
         * only.
         *
         * #### Notes
         * These are for user information and user interface hints only and should
         * not be relied on in code. A server may set these to empty strings if it
         * does not want to expose this information.
         *
         * Examples of appropriate use include displaying a help dialog for a user
         * listing the paths, or a tooltip in a filebrowser displaying the server
         * root. Examples of inappropriate use include using one of these paths in a
         * terminal command, generating code using these paths, or using one of
         * these paths in a request to the server (it would be better to write a
         * server extension to handle these cases).
         */
        readonly directories: {
            readonly appSettings: string;
            readonly schemas: string;
            readonly static: string;
            readonly templates: string;
            readonly themes: string;
            readonly userSettings: string;
            readonly serverRoot: string;
            readonly workspaces: string;
        };
    }
    /**
     * The application tree resolver token.
     *
     * #### Notes
     * Not all Jupyter front-end applications will have a tree resolver
     * implemented on the client-side. This token should not be required as a
     * dependency if it is possible to make it an optional dependency.
     */
    const ITreeResolver: Token<ITreeResolver>;
    /**
     * An interface for a front-end tree route resolver.
     */
    interface ITreeResolver {
        /**
         * A promise that resolves to the routed tree paths or null.
         */
        readonly paths: Promise<ITreeResolver.Paths>;
    }
    /**
     * A namespace for tree resolver types.
     */
    namespace ITreeResolver {
        /**
         * The browser and file paths if the tree resolver encountered and handled
         * a tree URL or `null` if not. Empty string paths should be ignored.
         */
        type Paths = {
            browser: string;
            file: string;
        } | null;
    }
}
/**
 * A namespace for the context menu override.
 */
export declare namespace JupyterFrontEndContextMenu {
    /**
     * An id for a private context-menu-info ersatz command.
     */
    const contextMenu = "__internal:context-menu-info";
}
