// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { Signal } from '@lumino/signaling';
/**
 * Abstract class implementing the search provider interface.
 */
export class SearchProvider {
    /**
     * Constructor
     *
     * @param widget The widget to search in
     */
    constructor(widget) {
        this.widget = widget;
        this._stateChanged = new Signal(this);
        this._filtersChanged = new Signal(this);
        this._disposed = false;
    }
    /**
     * Signal indicating that something in the search has changed, so the UI should update
     */
    get stateChanged() {
        return this._stateChanged;
    }
    /**
     * Signal indicating that filter definition changed.
     */
    get filtersChanged() {
        return this._filtersChanged;
    }
    /**
     * The current index of the selected match.
     */
    get currentMatchIndex() {
        return null;
    }
    /**
     * Whether the search provider is disposed or not.
     */
    get isDisposed() {
        return this._disposed;
    }
    /**
     * The number of matches.
     */
    get matchesCount() {
        return null;
    }
    /**
     * Dispose of the resources held by the search provider.
     *
     * #### Notes
     * If the object's `dispose` method is called more than once, all
     * calls made after the first will be a no-op.
     *
     * #### Undefined Behavior
     * It is undefined behavior to use any functionality of the object
     * after it has been disposed unless otherwise explicitly noted.
     */
    dispose() {
        if (this._disposed) {
            return;
        }
        this._disposed = true;
        Signal.clearData(this);
    }
    /**
     * Get an initial query value if applicable so that it can be entered
     * into the search box as an initial query
     *
     * @returns Initial value used to populate the search box.
     */
    getInitialQuery() {
        return '';
    }
    /**
     * Get the filters for the given provider.
     *
     * @returns The filters.
     *
     * ### Notes
     * TODO For now it only supports boolean filters (represented with checkboxes)
     */
    getFilters() {
        return {};
    }
    /**
     * Utility for copying the letter case from old to new text.
     */
    static preserveCase(oldText, newText) {
        if (oldText.toUpperCase() === oldText) {
            return newText.toUpperCase();
        }
        if (oldText.toLowerCase() === oldText) {
            return newText.toLowerCase();
        }
        if (toSentenceCase(oldText) === oldText) {
            return toSentenceCase(newText);
        }
        return newText;
    }
}
/**
 * Capitalise first letter of provided word.
 */
function toSentenceCase([first = '', ...suffix]) {
    return first.toUpperCase() + '' + suffix.join('').toLowerCase();
}
//# sourceMappingURL=searchprovider.js.map