import { IRetroable } from '@lumino/algorithm';
/**
 * A generic doubly-linked list.
 */
export declare class LinkedList<T> implements Iterable<T>, IRetroable<T> {
    /**
     * Whether the list is empty.
     *
     * #### Complexity
     * Constant.
     */
    get isEmpty(): boolean;
    /**
     * The size of the list.
     *
     * #### Complexity
     * `O(1)`
     *
     * #### Notes
     * This is equivalent to `length`.
     */
    get size(): number;
    /**
     * The length of the list.
     *
     * #### Complexity
     * Constant.
     *
     * #### Notes
     * This is equivalent to `size`.
     *
     * This property is deprecated.
     */
    get length(): number;
    /**
     * The first value in the list.
     *
     * This is `undefined` if the list is empty.
     *
     * #### Complexity
     * Constant.
     */
    get first(): T | undefined;
    /**
     * The last value in the list.
     *
     * This is `undefined` if the list is empty.
     *
     * #### Complexity
     * Constant.
     */
    get last(): T | undefined;
    /**
     * The first node in the list.
     *
     * This is `null` if the list is empty.
     *
     * #### Complexity
     * Constant.
     */
    get firstNode(): LinkedList.INode<T> | null;
    /**
     * The last node in the list.
     *
     * This is `null` if the list is empty.
     *
     * #### Complexity
     * Constant.
     */
    get lastNode(): LinkedList.INode<T> | null;
    /**
     * Create an iterator over the values in the list.
     *
     * @returns A new iterator starting with the first value.
     *
     * #### Complexity
     * Constant.
     */
    [Symbol.iterator](): IterableIterator<T>;
    /**
     * Create a reverse iterator over the values in the list.
     *
     * @returns A new iterator starting with the last value.
     *
     * #### Complexity
     * Constant.
     */
    retro(): IterableIterator<T>;
    /**
     * Create an iterator over the nodes in the list.
     *
     * @returns A new iterator starting with the first node.
     *
     * #### Complexity
     * Constant.
     */
    nodes(): IterableIterator<LinkedList.INode<T>>;
    /**
     * Create a reverse iterator over the nodes in the list.
     *
     * @returns A new iterator starting with the last node.
     *
     * #### Complexity
     * Constant.
     */
    retroNodes(): IterableIterator<LinkedList.INode<T>>;
    /**
     * Assign new values to the list, replacing all current values.
     *
     * @param values - The values to assign to the list.
     *
     * #### Complexity
     * Linear.
     */
    assign(values: Iterable<T>): void;
    /**
     * Add a value to the end of the list.
     *
     * @param value - The value to add to the end of the list.
     *
     * #### Complexity
     * Constant.
     *
     * #### Notes
     * This is equivalent to `addLast`.
     */
    push(value: T): void;
    /**
     * Remove and return the value at the end of the list.
     *
     * @returns The removed value, or `undefined` if the list is empty.
     *
     * #### Complexity
     * Constant.
     *
     * #### Notes
     * This is equivalent to `removeLast`.
     */
    pop(): T | undefined;
    /**
     * Add a value to the beginning of the list.
     *
     * @param value - The value to add to the beginning of the list.
     *
     * #### Complexity
     * Constant.
     *
     * #### Notes
     * This is equivalent to `addFirst`.
     */
    shift(value: T): void;
    /**
     * Remove and return the value at the beginning of the list.
     *
     * @returns The removed value, or `undefined` if the list is empty.
     *
     * #### Complexity
     * Constant.
     *
     * #### Notes
     * This is equivalent to `removeFirst`.
     */
    unshift(): T | undefined;
    /**
     * Add a value to the beginning of the list.
     *
     * @param value - The value to add to the beginning of the list.
     *
     * @returns The list node which holds the value.
     *
     * #### Complexity
     * Constant.
     */
    addFirst(value: T): LinkedList.INode<T>;
    /**
     * Add a value to the end of the list.
     *
     * @param value - The value to add to the end of the list.
     *
     * @returns The list node which holds the value.
     *
     * #### Complexity
     * Constant.
     */
    addLast(value: T): LinkedList.INode<T>;
    /**
     * Insert a value before a specific node in the list.
     *
     * @param value - The value to insert before the reference node.
     *
     * @param ref - The reference node of interest. If this is `null`,
     *   the value will be added to the beginning of the list.
     *
     * @returns The list node which holds the value.
     *
     * #### Notes
     * The reference node must be owned by the list.
     *
     * #### Complexity
     * Constant.
     */
    insertBefore(value: T, ref: LinkedList.INode<T> | null): LinkedList.INode<T>;
    /**
     * Insert a value after a specific node in the list.
     *
     * @param value - The value to insert after the reference node.
     *
     * @param ref - The reference node of interest. If this is `null`,
     *   the value will be added to the end of the list.
     *
     * @returns The list node which holds the value.
     *
     * #### Notes
     * The reference node must be owned by the list.
     *
     * #### Complexity
     * Constant.
     */
    insertAfter(value: T, ref: LinkedList.INode<T> | null): LinkedList.INode<T>;
    /**
     * Remove and return the value at the beginning of the list.
     *
     * @returns The removed value, or `undefined` if the list is empty.
     *
     * #### Complexity
     * Constant.
     */
    removeFirst(): T | undefined;
    /**
     * Remove and return the value at the end of the list.
     *
     * @returns The removed value, or `undefined` if the list is empty.
     *
     * #### Complexity
     * Constant.
     */
    removeLast(): T | undefined;
    /**
     * Remove a specific node from the list.
     *
     * @param node - The node to remove from the list.
     *
     * #### Complexity
     * Constant.
     *
     * #### Notes
     * The node must be owned by the list.
     */
    removeNode(node: LinkedList.INode<T>): void;
    /**
     * Remove all values from the list.
     *
     * #### Complexity
     * Linear.
     */
    clear(): void;
    private _first;
    private _last;
    private _size;
}
/**
 * The namespace for the `LinkedList` class statics.
 */
export declare namespace LinkedList {
    /**
     * An object which represents a node in a linked list.
     *
     * #### Notes
     * User code will not create linked list nodes directly. Nodes
     * are created automatically when values are added to a list.
     */
    interface INode<T> {
        /**
         * The linked list which created and owns the node.
         *
         * This will be `null` when the node is removed from the list.
         */
        readonly list: LinkedList<T> | null;
        /**
         * The next node in the list.
         *
         * This will be `null` when the node is the last node in the list
         * or when the node is removed from the list.
         */
        readonly next: INode<T> | null;
        /**
         * The previous node in the list.
         *
         * This will be `null` when the node is the first node in the list
         * or when the node is removed from the list.
         */
        readonly prev: INode<T> | null;
        /**
         * The user value stored in the node.
         */
        readonly value: T;
    }
    /**
     * Create a linked list from an iterable of values.
     *
     * @param values - The iterable object of interest.
     *
     * @returns A new linked list initialized with the given values.
     *
     * #### Complexity
     * Linear.
     */
    function from<T>(values: Iterable<T>): LinkedList<T>;
}
