// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { IEditorMimeTypeService } from '@jupyterlab/codeeditor';
import { PathExt } from '@jupyterlab/coreutils';
/**
 * The mime type service for CodeMirror.
 */
export class CodeMirrorMimeTypeService {
    constructor(languages) {
        this.languages = languages;
    }
    /**
     * Returns a mime type for the given language info.
     *
     * #### Notes
     * If a mime type cannot be found returns the default mime type `text/plain`, never `null`.
     * There may be more than one mime type, but only the first one will be returned.
     * To access all mime types, use `IEditorLanguageRegistry` instead.
     */
    getMimeTypeByLanguage(info) {
        var _a;
        const ext = info.file_extension || '';
        const mode = this.languages.findBest(info.codemirror_mode || {
            mimetype: info.mimetype,
            name: info.name,
            ext: [ext.split('.').slice(-1)[0]]
        });
        return mode
            ? Array.isArray(mode.mime)
                ? (_a = mode.mime[0]) !== null && _a !== void 0 ? _a : IEditorMimeTypeService.defaultMimeType
                : mode.mime
            : IEditorMimeTypeService.defaultMimeType;
    }
    /**
     * Returns a mime type for the given file path.
     *
     * #### Notes
     * If a mime type cannot be found returns the default mime type `text/plain`, never `null`.
     * There may be more than one mime type, but only the first one will be returned.
     * To access all mime types, use `IEditorLanguageRegistry` instead.
     */
    getMimeTypeByFilePath(path) {
        var _a;
        const ext = PathExt.extname(path);
        if (ext === '.ipy') {
            return 'text/x-python';
        }
        else if (ext === '.md') {
            return 'text/x-ipythongfm';
        }
        const mode = this.languages.findByFileName(path);
        return mode
            ? Array.isArray(mode.mime)
                ? (_a = mode.mime[0]) !== null && _a !== void 0 ? _a : IEditorMimeTypeService.defaultMimeType
                : mode.mime
            : IEditorMimeTypeService.defaultMimeType;
    }
}
//# sourceMappingURL=mimetype.js.map