---
title: Configuration File Reference
---

# {% $markdoc.frontmatter.title %}

Nixpacks supports specifying build configuration in a `nixpacks.toml` or `nixpacks.json` file. The config will automatically be used if one of these files is found in the app root. Otherwise, the file can be specified with the `--config <file>` flag or with the `NIXPACKS_CONFIG_FILE` environment variable.

The contents of this file can contain a full build plan, which means that every aspect of the build can be customized. An example config looks something like:

```toml
# nixpacks.toml

[phases.setup]
nixPkgs = ['...', 'cowsay']

[phases.build]
cmds = ['yarn run server:build']

[start]
cmd = 'yarn run server:start'
```

The file-based build plan is merged with the build plan generated by the provider, as well as the configuration from the environment variables and CLI flags. By default values are overridden when merged if they are not null. The configuration priority is

- Provider (lowest)
- File
- Environment
- CLI

## Array Extending

The default behaviour when merging build plans is for a non-null value of higher priority to override a lower priority value. However, you can use the `"..."` special syntax to _extend_ the values in an array. For example:

```toml
# Build plan from the provider
[phases.setup]
nixPkgs = ['nodejs', 'yarn']

# nixpacks.toml
[phases.setup]
nixPkgs = ['...', 'cowsay']

# Merged plan
[phases.setup]
nixPkgs = ['nodejs', 'yarn', 'cowsay']
```

The `"..."` represents a hole that will be populated by the values from plan that is merged into.

---

## Providers

Specify the providers that you want to run on the build.

```toml
providers = ['...', 'python']
```

## Build image

The image to use as the base when building the application.

```toml
buildImage = 'ghcr.io/railwayapp/nixpacks:latest'
```

## Variables

Key-value pairs of variables to include in the final image.

```toml
[variables]
NODE_ENV = 'production'
HELLO = 'world'
```

## Static assets

Files that are copied into the `/assets` directory of the image.

```toml
[staticAssets]
myFile = '''
  asdfasdfasdf
'''
```

## Phases

The phases specify exactly how the application is built and packaged into an image. Each phase can depend on a list of other phases and the ordering is resolved when the `Dockerfile` is automatically generated and run. The phases are typically defined as

- `setup`: Install the Nix/Apt packages (e.g. NodeJS, Yarn)
- `install`: Install the language/framework dependencies (e.g. Yarn install)
- `build`: Build the application (e.g. Yarn build)

However, there is no restriction on the names of phases or what they are named.

```toml
[phases.setup]
# ...

[phases.install]
# ...

[phases.build]
# ...
```

### Commands

Array of commands to run.

```toml
[phase.name]
  cmds = ['cmd1', 'cmd2']
```

### Nix packages

Nix packages to install. Available packages can be found at [search.nixos.org](https://search.nixos.org/packages?channel=unstable).

```toml
[phase.name]
  nixPkgs = ['cowsay']
```

### Nix libraries

Nix packages to be made available through the `LD_LIBRARY_PATH` environment variables. The paths to each packages library files are appended.

```toml
[phase.name]
  nixLibs = ['zlib']
```

### Nix overlays

[Nix overlays](https://wiki.nixos.org/wiki/Overlays) to use as alternate package sources.

```toml
[phase.name]
  nixOverlays = ['https://github.com/oxalica/rust-overlay/archive/master.tar.gz']
```

### Nixpkgs archive

Specific version of the Nixpkgs archive to use. By default all builds are built using the version defined [here](https://github.com/railwayapp/nixpacks/blob/2d16cd938c95411db4a0c56b81bf7b558252af7b/src/nixpacks/nix/mod.rs#L11). But this value can be overridden to install Nix packages from an older or newer archive.

```toml
[phase.name]
  nixpkgsArchive = '5148520bfab61f99fd25fb9ff7bfbb50dad3c9db'
```

### Apt packages

List of packages to install with `apt-get`

```toml
[phase.name]
  aptPkgs = ['wget']
```

### Phase dependencies

List of phases that this phase must run after.

```toml
[phase.name]
  dependsOn = ['install']
```

### Cache directories

Directories to cache for this phase. Cached directories do not appear in the final build. See [/caching](/docs/configuration/caching) for more information.

```toml
[phase.name]
  cacheDirectories = ['node_modules/.cache']
```

### Included files

Files to **only** make available when running this phase. If no array is specified, then all files are copied into the image before running the commands. This can be useful when optimizing the Docker layer cache.

```toml
[phase.name]
  onlyIncludeFiles = ['package.json']
```

_Note: Files included from previous phases will also be available._

### Paths

Paths to append to the `PATH` environment variable.

```toml
[phase.name]
  paths = ['/app/node_modules/.bin']
```

## Start Phase

This configures how a container created from the image will start.

### Command

The command to run when starting the container.

```toml
[start]
  cmd = "yarn run start"
```

### Run image

The runtime image to use. If not specified, the same build image will be used.

```toml
[start]
  runImage = 'debian:bullseye-slim'
```

### Included files

Must be used in combination with `runImage`. The only files that should be copied over to the run image. If no value is specified, the entire app directory is copied over.

```toml
[start]
  onlyIncludeFiles = ['./bin/rust-custom-version']
```
